import { supabase, supabaseAdmin, TABLES } from '@/config/supabase';
import { 
  DebugPingResponse,
  SupabaseConnectionTest,
  UserDebugInfo,
  FeatureCheckRequest,
  FeatureCheckResponse,
  SubscriptionCheckRequest,
  SubscriptionCheckResponse,
  AILogRequest,
  AILogResponse,
  SystemStatus,
  CacheDebugInfo,
  DebugLogEntry
} from './types';
import { aiService } from '@/modules/ai/service';
import { subscriptionService } from '@/modules/subscriptions/service';

export const debugService = {
  // GET /debug/ping
  async ping(): Promise<DebugPingResponse> {
    return {
      status: 'ok',
      timestamp: Math.floor(Date.now() / 1000)
    };
  },

  // GET /debug/supabase
  async testSupabaseConnection(): Promise<SupabaseConnectionTest> {
    try {
      const startTime = Date.now();
      
      // Test basic connection
      const { data: users, error: usersError } = await supabaseAdmin
        .from(TABLES.USERS)
        .select('id')
        .limit(1);

      if (usersError) {
        return {
          connected: false,
          tables: [],
          error: usersError.message
        };
      }

      // Test key tables
      const keyTables = [
        TABLES.USERS,
        TABLES.VENDORS,
        TABLES.SUBSCRIPTION_PLANS,
        TABLES.VENDOR_SUBSCRIPTIONS,
        TABLES.FEATURES,
        TABLES.CUSTOMERS,
        TABLES.INVOICES,
        TABLES.ORDERS
      ];

      const accessibleTables: string[] = [];
      
      for (const table of keyTables) {
        try {
          const { error } = await supabaseAdmin
            .from(table)
            .select('id')
            .limit(1);
          
          if (!error) {
            accessibleTables.push(table);
          }
        } catch (tableError) {
          console.log(`Table ${table} not accessible:`, tableError);
        }
      }

      const responseTime = Date.now() - startTime;

      return {
        connected: true,
        tables: accessibleTables,
        error: accessibleTables.length < keyTables.length ? 'Some tables not accessible' : undefined
      };
    } catch (error: any) {
      return {
        connected: false,
        tables: [],
        error: error.message
      };
    }
  },

  // GET /debug/user/:user_id
  async getUserDebugInfo(userId: string): Promise<{ data?: UserDebugInfo; error?: string }> {
    try {
      // Get user info
      const { data: user, error: userError } = await supabaseAdmin
        .from(TABLES.USERS)
        .select('*')
        .eq('id', userId)
        .single();

      if (userError) throw new Error('User not found');

      // Get vendor info
      const { data: vendor } = await supabaseAdmin
        .from(TABLES.VENDORS)
        .select('*')
        .eq('user_id', userId)
        .single();

      // Get subscription info
      const { data: subscription } = await supabaseAdmin
        .from(TABLES.VENDOR_SUBSCRIPTIONS)
        .select(`
          *,
          subscription_plans:plan_id(*)
        `)
        .eq('vendor_id', vendor?.id)
        .eq('status', 'active')
        .single();

      // Get features
      const features = vendor ? await subscriptionService.getAvailableFeatures(userId) : [];

      // Calculate trial state
      const trialState = {
        is_trial: false,
        days_remaining: 0
      };

      if (subscription?.trial_end) {
        const trialEnd = new Date(subscription.trial_end);
        const now = new Date();
        const daysRemaining = Math.ceil((trialEnd.getTime() - now.getTime()) / (1000 * 60 * 60 * 24));
        
        trialState.is_trial = daysRemaining > 0;
        trialState.days_remaining = Math.max(0, daysRemaining);
      }

      return {
        data: {
          user_id: userId,
          email: user.email,
          role: user.role,
          vendor: vendor ? {
            id: vendor.id,
            name: vendor.name,
            status: vendor.status
          } : undefined,
          subscription: subscription ? {
            plan: subscription.subscription_plans?.name || 'Unknown',
            status: subscription.status,
            trial_end: subscription.trial_end
          } : undefined,
          features,
          trial_state: trialState
        }
      };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // POST /debug/feature-check
  async checkFeature(request: FeatureCheckRequest): Promise<{ data?: FeatureCheckResponse; error?: string }> {
    try {
      const { user_id, feature } = request;

      // Get user's features
      const features = await subscriptionService.getAvailableFeatures(user_id);
      const enabled = features.includes(feature);

      // Get subscription info
      const { data: vendor } = await supabaseAdmin
        .from(TABLES.VENDORS)
        .select('id')
        .eq('user_id', user_id)
        .single();

      let source: 'plan' | 'trial' | 'admin_override' = 'plan';
      let plan: string | undefined;
      let trialDaysRemaining: number | undefined;

      if (vendor) {
        const { data: subscription } = await supabaseAdmin
          .from(TABLES.VENDOR_SUBSCRIPTIONS)
          .select(`
            *,
            subscription_plans:plan_id(*)
          `)
          .eq('vendor_id', vendor.id)
          .eq('status', 'active')
          .single();

        if (subscription) {
          plan = subscription.subscription_plans?.name;
          
          if (subscription.trial_end) {
            const trialEnd = new Date(subscription.trial_end);
            const now = new Date();
            const daysRemaining = Math.ceil((trialEnd.getTime() - now.getTime()) / (1000 * 60 * 60 * 24));
            
            if (daysRemaining > 0) {
              source = 'trial';
              trialDaysRemaining = daysRemaining;
            }
          }
        }
      }

      return {
        data: {
          enabled,
          source,
          plan,
          trial_days_remaining: trialDaysRemaining
        }
      };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // POST /debug/subscription-check
  async checkSubscription(request: SubscriptionCheckRequest): Promise<{ data?: SubscriptionCheckResponse; error?: string }> {
    try {
      const { vendor_id } = request;

      // Get vendor info
      const { data: vendor, error: vendorError } = await supabaseAdmin
        .from(TABLES.VENDORS)
        .select('*')
        .eq('id', vendor_id)
        .single();

      if (vendorError) throw new Error('Vendor not found');

      // Get subscription
      const { data: subscription } = await supabaseAdmin
        .from(TABLES.VENDOR_SUBSCRIPTIONS)
        .select(`
          *,
          subscription_plans:plan_id(*)
        `)
        .eq('vendor_id', vendor_id)
        .eq('status', 'active')
        .single();

      // Get features
      const features = await subscriptionService.getAvailableFeatures(vendor.user_id);

      // Calculate trial state
      const trial = {
        is_trial: false,
        days_remaining: 0
      };

      if (subscription?.trial_end) {
        const trialEnd = new Date(subscription.trial_end);
        const now = new Date();
        const daysRemaining = Math.ceil((trialEnd.getTime() - now.getTime()) / (1000 * 60 * 60 * 24));
        
        trial.is_trial = daysRemaining > 0;
        trial.days_remaining = Math.max(0, daysRemaining);
      }

      return {
        data: {
          vendor_id,
          plan: subscription?.subscription_plans?.name || 'No Plan',
          status: subscription?.status || 'No Subscription',
          trial,
          features,
          limits: {
            websites: subscription?.subscription_plans?.website_limit,
            products: subscription?.subscription_plans?.product_limit,
            edits: subscription?.subscription_plans?.edit_limit
          }
        }
      };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // POST /debug/ai-log
  async testAILog(request: AILogRequest): Promise<{ data?: AILogResponse; error?: string }> {
    try {
      const { message, source_module, user_id } = request;
      const startTime = Date.now();
      const traceId = `debug_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

      // Test AI service
      const aiResponse = await aiService.generateGeneralAssistant(message, source_module);
      const latency = Date.now() - startTime;

      // Log to database
      try {
        await supabaseAdmin
          .from(TABLES.AI_LOGS)
          .insert({
            user_id,
            message,
            response: aiResponse,
            source_module,
            trace_id: traceId,
            latency,
            success: true
          });
      } catch (logError) {
        console.log('Failed to log AI request:', logError);
      }

      return {
        data: {
          ai_response: aiResponse,
          trace_id: traceId,
          latency,
          success: true
        }
      };
    } catch (error: any) {
      return {
        data: {
          ai_response: '',
          trace_id: `debug_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
          latency: 0,
          success: false,
          error: error.message
        }
      };
    }
  },

  // GET /debug/system-status
  async getSystemStatus(): Promise<{ data?: SystemStatus; error?: string }> {
    try {
      const startTime = Date.now();
      
      // Test API response time
      const apiResponseTime = Date.now() - startTime;
      
      // Test Supabase
      const supabaseTest = await this.testSupabaseConnection();
      
      // Test AI service
      const aiStartTime = Date.now();
      let aiStatus: 'available' | 'unavailable' | 'error' = 'available';
      let aiResponseTime = 0;
      
      try {
        await aiService.generateGeneralAssistant('test', 'debug');
        aiResponseTime = Date.now() - aiStartTime;
      } catch (aiError) {
        aiStatus = 'error';
        aiResponseTime = Date.now() - aiStartTime;
      }

      // Test feature gates (simulate)
      const featureGatesStatus = 'working';
      const cachedFeaturesCount = 0; // Would be actual cache count

      // Simulate queue status
      const queueStatus = 'idle';
      const pendingJobs = 0;

      // Determine overall status
      let overallStatus: 'healthy' | 'degraded' | 'down' = 'healthy';
      
      if (supabaseTest.connected === false || aiStatus === 'error') {
        overallStatus = 'down';
      } else if (aiStatus === 'unavailable' || supabaseTest.error) {
        overallStatus = 'degraded';
      }

      return {
        data: {
          api: {
            status: 'healthy',
            response_time: apiResponseTime
          },
          supabase: {
            status: supabaseTest.connected ? 'connected' : 'disconnected',
            tables_accessible: supabaseTest.tables.length
          },
          ai_service: {
            status: aiStatus,
            response_time: aiResponseTime
          },
          feature_gates: {
            status: featureGatesStatus,
            cached_features: cachedFeaturesCount
          },
          queue: {
            status: queueStatus,
            pending_jobs: pendingJobs
          },
          overall_status: overallStatus,
          timestamp: new Date().toISOString()
        }
      };
    } catch (error: any) {
      return { error: error.message };
    }
  },

  // GET /debug/cache
  async getCacheDebugInfo(userId: string): Promise<{ data?: CacheDebugInfo; error?: string }> {
    try {
      // Get user's features (this would normally be cached)
      const features = await subscriptionService.getAvailableFeatures(userId);
      
      return {
        data: {
          user_id: userId,
          cached_features: features,
          cache_timestamp: new Date().toISOString(),
          cache_ttl: 3600 // 1 hour TTL
        }
      };
    } catch (error: any) {
      return { error: error.message };
    }
  }
}; 